<?php
/**
 * See the enclosed file COPYING for license information (LGPL). If you
 * did not receive this file, see http://www.fsf.org/copyleft/lgpl.html.
 *
 * $Horde: presentations/nyphpcon-2006-06/Helpers/Form.php,v 1.1 2007/01/26 21:20:54 chuck Exp $
 *
 * @package Horde_RDO
 */

/** Horde_Form */
require_once 'Horde/Form.php';

/** Horde_Form_Renderer */
require_once 'Horde/Form/Renderer.php';

/**
 * The Horde_Form_Helper:: class provides autogeneration extension to
 * Horde_Form used for generating create and update data with various
 * backends.
 *
 * @author  Duck <duck@obala.net>
 * @since   Horde 4.0
 * @package Horde_RDO
 */
class Horde_Form_Helper extends Horde_Form {

    /**
     * The primary key(s) of the resource.
     *
     * @var string
     */
    protected $primaryKey;

    /**
     * RDO_Mapper object that we display.
     *
     * @var Horde_RDO_Mapper $mapper
     */
    protected $mapper;

    /**
     * Extends Horde_Form for general purposes and prepare initial form.
     */
    public function __construct($mapper, $vars, $title = '', $name = null, $params = array())
    {
        $this->mapper = $mapper;

        parent::Horde_Form($vars, $title, $name);

        $action = $vars->action;
        if (isset($params['action'])) {
            $action = $params['action'];
        }

        $this->addHidden('', 'action', 'text', true);
        $this->addHidden('', 'table', 'text', true);
        $this->addHidden('', 'what2process', 'text', true);

        $this->getPrimaryKey();
        foreach ($this->getFields($action) as $key) {
            if ($key == 'created' || $key == 'updated') {
                continue;
            }

            $params = $this->formMeta($action, $key);
            $this->_currentSection = $params['section'];

            switch ($action) {
            case 'create':
                if ($params['readonly']) {
                    continue 2;
                }
                break;

            case 'update':
                if (in_array($key, $this->primaryKey)) {
                    $this->addHidden('', $key, $params['type'], true);
                    $params['readonly'] = true;
                }
                break;

            case 'search':
                $params['readonly'] = false;
                $params['required'] = false;
                break;
            }

            if ($params['hidden']) {
                $this->addHidden('', $key, $params['type'], $params['required']);
                continue;
            }

            if ($action == 'search' && !$params['hidden']) {
                $v = $this->addVariable($params['humanName'], 'cases_' . $key , 'enum',
                                        true, false, 'cases', array(array('=' => ' = ',
                                                                          '>' => ' > ',
                                                                          '<' => ' < ',
                                                                          '<>' => ' <> ')));
            }

            $v = $this->addVariable($params['humanName'], $key, $params['type'],
                                    $params['required'], $params['readonly'],
                                    $params['description'], $params['params']);

            if (!empty($params['help'])) {
                $v->setHelp($params['help']);
            }
        }

        if (isset($vars->action)) {
            $this->addHidden('', 'action', 'text', true);
        }

        switch ($action) {
        case 'search':
            $this->_submit = _("Search");
            $vars->set('action', 'search');
            break;

        case 'create':
            $this->_submit = _("Create");
            $vars->set('action', 'create');
            break;

        case 'update':
            $this->_submit = _("Update");
            $this->_reset = _("Reset");
        }
    }

    /**
     * Map from columns to Horde_Form types and attributes:
     * Create values like Horde_Form::addVariable parameters
     *   - humanName
     *   - type
     *   - required
     *   - readonly
     *   - description
     *   - params
     */
    protected function formMeta($action, $column, $key = false)
    {
        static $map;

        if ($map === null) {
            $map = $this->formMetaData($action);
            if (isset($map['__sections'])) {
                foreach ($map['__sections'] as $section => $value) {
                    $this->setSection($section, $value['desc'], $value['image'], $value['expanded']);
                }
                unset($map['__sections']);
            }

            foreach ($this->getFields($action) as $id) {
                if (!isset($map[$id])) {
                    $map[$id] = array();
                }
                if (!isset($map[$id]['hidden'])) {
                    $map[$id]['hidden'] = false;
                }
                if (!isset($map[$id]['humanName'])) {
                    $map[$id]['humanName'] = $id;
                }
                if (!isset($map[$id]['section'])) {
                    $map[$id]['section'] = '__base';
                }
                if (!isset($map[$id]['readonly'])) {
                    $map[$id]['readonly'] = false;
                }
                if (!isset($map[$id]['params'])) {
                    $map[$id]['params'] = array();
                }
                if (!isset($map[$id]['description'])) {
                    $map[$id]['description'] = '';
                }
                if (!isset($map[$id]['required'])) {
                    $map[$id]['required'] = in_array($id, $this->primaryKey);
                }
                if (in_array($id, $this->primaryKey)) {
                    $map[$id]['readonly'] = true;
                }

                if (!isset($map[$id]['type'])) {
                    $map[$id]['type'] = 'text';
                } else {
                    // Trim additional parameters like decimal(10,2)
                    if (strpos($map[$id]['type'], '(') !== false) {
                        $map[$id]['type'] = substr($map[$id]['type'], 0, strpos($map[$id]['type'], '('));
                    }

                    switch ($map[$id]['type']) {
                    // Date types
                    case 'date':
                        $map[$id]['type'] = 'monthdayyear';
                        $map[$id]['params'] = array('', '', true, '%Y-%m-%d');
                        break;

                    // Number types
                    case 'number':
                    case 'decimal':
                    case 'real':
                        $map[$id]['type'] = 'number';
                        break;

                    // Types to not muck with
                    case 'int':
                    case 'longtext':
                    case 'enum':
                        break;

                    case 'text':
                        $map[$id]['type'] = 'longtext';
                        break;

                    default:
                        $map[$id]['type'] = 'text';
                        break;
                    }
                }
            }
        }

        if ($key) {
            return $map[$column][$key];
        } else {
            return $map[$column];
        }
    }

    /**
     * Return the selected object or false if none.
     */
    public function getSelected()
    {
        static $params;

        if (is_null($params)) {
            foreach ($this->primaryKey as $key) {
                if ($this->_vars->$key) {
                    $params[$key] = $this->_vars->$key;
                }
            }
        }

        return $this->mapper->find(Horde_RDO::FIND_FIRST, $params);
    }

    /**
     * Return the form meta data
     */
    public function formMetaData($action)
    {
        if (method_exists($this->mapper, 'formMeta')) {
            return $this->mapper->formMeta($action);
        } else {
            return $this->mapper->getModel()->getFields();
        }
    }

    /**
     * Array of field name.
     */
    protected function getFields($action)
    {
        static $fields;

        if ($fields !== null) {
            return $fields;
        }

        if (method_exists($this->mapper, 'formFields')) {
            $fields = $this->mapper->formFields($action);
        } else {
            $fields = $this->mapper->getModel()->listFields();
        }

        return $fields;
    }

    /**
     * Get primary key.
     */
    protected function getPrimaryKey()
    {
        if ($this->primaryKey !== null) {
            return $this->primaryKey;
        }

        if (method_exists($this->mapper, 'getPrimaryKey')) {
            $this->primaryKey = $this->mapper->getPrimaryKey();
        } else {
            $this->primaryKey = array($this->mapper->getModel()->key);
        }
    }

    /**
     * Delect selected record.
     */
    public function delete()
    {
        return $this->getSelected()->delete();
    }

    /**
     * Create a new record.
     */
    public function create($params)
    {
        foreach ($params as $key => $value) {
            $meta = $this->formMeta('update', $key);
            if ($meta['type'] == 'set') {
                $params[$key] = implode('|', $value);
            }
        }

        return $this->mapper->create($params);
    }

    /**
     * Update selected record.
     */
    public function update($params)
    {
        foreach ($params as $key => $value) {
            $meta = $this->formMeta('update', $key);
            if ($meta['type'] == 'set') {
                $params[$key] = implode('|', $value);
            }
        }

        $selected = $this->getSelected();
        foreach ($params as $key => $value) {
            $selected->$key = $value;
        }

        return $selected->save();
    }

}
