<?php
/**
 * See the enclosed file COPYING for license information (LGPL). If you
 * did not receive this file, see http://www.fsf.org/copyleft/lgpl.html.
 *
 * $Horde: presentations/nyphpcon-2006-06/Helpers/Table.php,v 1.2 2007/01/29 16:41:36 chuck Exp $
 *
 * @package Horde_RDO
 */

/** Horde_Template */
require_once 'Horde/Template.php';

/** Horde_RDO_Lens */
require_once dirname(__FILE__) . '/Lens.php';

/**
 * The Horde_Table_Helper class provides an RDO extension to Horde_Template
 * used to generate browsing lists for different backends.
 *
 * @author  Duck <duck@obala.net>
 * @since   Horde 4.0
 * @package Horde_RDO
 */
abstract class Horde_Table_Helper extends Horde_Template {

    /**
     * Parameters for this Template instance.
     *
     * @var array
     */
    protected $params = array();

    /**
     * Attempts to return a concrete Horde_Table_Helper instance based on
     * $driver.
     *
     * @param string $driver  The type of concrete Horde_Table_Helper subclass
     *                        to return.
     * @param array $params   A hash containing any additional configuration or
     *                        connection parameters a subclass might need.
     * @param mixed $object   Object to process with this driver
     *
     * @return Horde_Table_Helper  The newly created Horde_Table_Helper
     *                             instance, or false on an error.
     */
    public static function factory($driver, $params = array(), $object)
    {
        $driver = basename($driver);
        $class = 'Horde_Table_Helper_' . $driver;
        if (!class_exists($class)) {
            include dirname(__FILE__) . '/Table/' . $driver . '.php';
        }
        if (class_exists($class)) {
            return new $class($params, $object);
        } else {
            return PEAR::raiseError('Class definition of ' . $class . ' not found.');
        }
    }

    /**
     * Constructor
     *
     * @param array $params Template defaults.
     */
    public function __construct($params = array())
    {
        $defaults = array('delete' => true,
                          'update' => true,
                          'columns' => null,
                          'page' => 0,
                          'name' => '',
                          'id' => '',
                          'url' => Horde::selfUrl(),
                          'img_dir' => $GLOBALS['registry']->getImageDir('horde'),
                          'sort' => array(),
                          'filter' => array(),
                          );

        $this->params = array_merge($defaults, $params);
        if (!is_array($this->params['sort'])) {
            $this->params['sort'] = array($this->params['sort']);
        }

        parent::Horde_Template();
        $this->setOption('gettext', true);
    }

    /**
     * Fetch template.
     *
     * @param string $template Template path.
     */
    public function fetch($template = null)
    {
        if ($template === null) {
            $template = $this->getTemplateFile();
        }
        if ($template) {
            return parent::fetch($template);
        }
        return parent::parse($this->getTemplate());
    }

    /**
     * Fill up table data.
     */
    public function fill()
    {
        $this->set('rows', $this->getRows());

        if (empty($this->_scalars['img_dir'])) {
            $this->set('img_dir', $this->params['img_dir']);
        }

        if (empty($this->_scalars['url'])) {
            $this->set('url', $this->params['url']);
        }
    }

    /**
     * Get template path if it exists.
     *
     * @return string Template filename.
     */
    protected function getTemplateFile()
    {
        $filename = $GLOBALS['registry']->get('templates') . DIRECTORY_SEPARATOR . $this->params['name'] . '.html';
        if (file_exists($filename)) {
            return $filename;
        }

        $filename = $GLOBALS['registry']->get('templates', 'horde') . DIRECTORY_SEPARATOR . $this->params['name'] . '.html';
        if (file_exists($filename)) {
            return $filename;
        }

        return false;
    }

    /**
     * Create template content.
     *
     * @return string $content
     */
    public function getTemplate()
    {
        $url = '<tag:url />';
        $columns = $this->listFields();

        $keys = array();
        $primaryKeys = $this->getPrimaryKey();
        foreach ($primaryKeys as $key) {
            $keys[] = $key . '=<tag:rows.' . $key . ' />';
        }
        $keys = implode('&', $keys);

        $content = '<table id="' . $this->params['id'] . '" class="striped sortable">' . "\n" .
            '<thead>' . "\n" . '<tr>' . "\n";
        if ($this->params['update'] || $this->params['delete']) {
            $content .= '<th class="nosort"><gettext>Actions</gettext></th>' . "\n";
        }
        foreach ($columns as $key => $name) {
            if (in_array($key, $this->params['sort'])) {
                $content .= '<th class="sortdown">' . htmlspecialchars($name) . '</th>' . "\n";
            } else {
                $content .= '<th>' . htmlspecialchars($name) . '</th>' . "\n";
            }
        }
        $content .= '</tr>' . "\n" . '</thead>' . "\n" . '<tbody>' .
            '<loop:rows><tr>' . "\n";

        /* Actions. */
        if ($this->params['update'] || $this->params['delete']) {
            $content .= '<td class="nowrap">' . "\n";
            if ($this->params['update']) {
                $content .= '<a class="update" href="' . $url . '&action=update&' . $keys . '">' .
                    '<img src="<tag:img_dir />/edit.png" alt="<gettext>Edit</gettext>" title="<gettext>Edit</gettext>" /></a> ' . "\n";
            }
            if ($this->params['delete']) {
                $content .= '<a class="delete" href="' . $url . '&action=delete&' . $keys . '">' .
                    '<img src="<tag:img_dir />/delete.png" alt="<gettext>Delete</gettext>" title="<gettext>Delete</gettext>" /></a> ' . "\n";
            }
            $content .= '</td>' . "\n";
        }

        foreach ($columns as $key => $name) {
            $content .= '<td><tag:rows.' . $key . ' /></td>';
        }

        return $content . '</tr>' . "\n" . '</loop:rows>' . "\n" . '</tbody></table>';
    }

    /**
     * Get rows
     */
    abstract public function getRows();

    /**
     * Count rows
     *
     * @param array $filter filter/search rows
     */
    abstract public function count();

    /**
     * Return column names.
     */
    abstract protected function getFields();

    /**
     * Return column names.
     */
    abstract protected function listFields();

    /**
     * Get primary key.
     */
    abstract protected function getPrimaryKey();

}

/**
 * @package Horde_RDO
 */
class Horde_RDO_Table_Lens extends Horde_RDO_Lens {

    /**
     */
    private $date_format = '%x';

    /**
     */
    private $time_format = 'G:i';

    /**
     */
    public function __construct()
    {
        $this->date_format = $GLOBALS['prefs']->getValue('date_format');
        $this->time_format = $GLOBALS['prefs']->getValue('twentyFour') ? 'G:i' : 'g:ia';
    }

    /**
     */
    public function decorate($target)
    {
        if (!is_object($target)) {
            $target = (object)$target;
        }
        return parent::decorate($target);
    }

    /**
     */
    public function __get($key)
    {
        $value = parent::__get($key);
        if ($key == 'updated' || $key == 'created') {
            return strftime($this->date_format, $value) . ' ' . date($this->time_format , $value);
        }
        return htmlspecialchars($value);
    }

}

/**
 * The Horde_Table_Helper_RDO:: class provides RDO extension to Horde_Table_Helper
 * used for generating browsing lists for RDO instanses
 *
 * See the enclosed file COPYING for license information (LGPL). If you
 * did not receive this file, see http://www.fsf.org/copyleft/lgpl.html.
 *
 * @author  Duck <duck@obala.net>
 * @since   Horde 4.0
 * @package Horde_RDO
 */
class Horde_Table_Helper_RDO extends Horde_Table_Helper {

    /**
     * Mapper object that we display.
     *
     * @var Horde_RDO_Mapper $object
     */
    protected $object;

    /**
     * Constructor
     *
     * @param Horde_RDO_Mapper $object Mapper instance.
     * @param array $params Template defaults.
     */
    public function __construct($params = array(), $object)
    {
        $this->object = $object;
        $params['name'] = $object->getModel()->table;
        $params['id'] = $object->getModel()->table;

        parent::__construct($params);
    }

    /**
     * Get RDO members.
     */
    public function getRows()
    {
        $query = $this->prepareQuery();
        if (!empty($this->params['perpage'])) {
            $limit = $this->params['perpage'];
            $offset = $this->params['page'] * $this->params['perpage'];
            $query->limit($limit, $offset);
        }

        $rows = new Horde_RDO_List($query, Horde_RDO::FIND_ALL);

        if (!empty($this->params['decorator'])) {
            $decorator = $this->params['decorator'];
            $rows = new Horde_RDO_Decorating_Iterator($rows, new $decorator());
        }

        return $rows;
    }

    /**
     * Count RDO members.
     *
     * @param array $filter filter/search rows
     */
    public function count()
    {
        return $this->object->count($this->prepareQuery());
    }

    /**
     * Prepares RDO query.
     */
    private function prepareQuery()
    {
        $query = new Horde_RDO_Query($this->object);

        if (!empty($this->params['filter'])) {
            foreach ($this->params['filter'] as $key => $val) {
                if (is_array($val)) {
                    $query->addTest($val['field'], $val['test'], $val['value']);
                } else {
                    $query->addTest($key, '=', $val);
                }
            }
        }

        if (!empty($this->params['sort'])) {
            if (!is_array($this->params['sort'])) {
                $this->params['sort'] = array($this->params['sort']);
            }
            foreach ($this->params['sort'] as $sort) {
                if (is_array($sort)) {
                    $query->addSort($sort[0], $sort[1]);
                } else {
                    $query->addSort($sort);
                }
            }
        }

        return $query;
    }

    /**
     * Return field information.
     */
    protected function getFields()
    {
        if ($this->params['columns']) {
            return $this->params['columns'];
        } else {
            return $this->object->getModel()->listFields();
        }
    }

    /**
     * Return field names.
     */
    protected function listFields()
    {
        $meta = array();
        if (method_exists($this->object, 'formMeta')) {
            $meta = $this->object->formMeta('table');
        }

        $columns = array();

        $keys = $this->getFields();
        foreach ($keys as $key) {
            if (isset($meta[$key]['humanName'])) {
                $columns[$key] = $meta[$key]['humanName'];
            } elseif ($key == 'created') {
                $columns[$key] = 'Created';
            } elseif ($key == 'updated') {
                $columns[$key] = 'Updated';
            } else {
                $columns[$key] = $key;
            }
        }

        return $columns;
    }

    /**
     * Get primary key.
     */
    protected function getPrimaryKey()
    {
        if (method_exists($this->object, 'getPrimaryKey')) {
            $keys = $this->object->getPrimaryKey();
        } else {
            $keys = array($this->object->getModel()->key);
        }

        if (!$keys) {
            $keys = $this->getFields();
        }

        return $keys;
    }

}
